<?php
namespace CustomFieldsManager\Core\Admin;

require "blog-manager-export.php";
require "blog-manager-import.php";
require "blog_categories_mapping_form.php";


if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

class Admin
{

	private $menus = [];

	/**
	 * Get module name.
	 *
	 * Retrieve the module name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Module name.
	 */
	public function get_name()
	{
		return 'admin';
	}

	/**
	 * Get assets base url
	 *
	 * @since 1.0.0
	 * @access protected
	 *
	 * @return string
	 */
	protected function get_assets_base_url()
	{
		return CUSTOM_FIELDS_MANAGER_URL . 'assets/';
	}

	/**
	 * Get assets url.
	 *
	 * @since 1.0.0
	 * @access protected
	 *
	 * @param string $file_name
	 * @param string $file_extension
	 *
	 * @return string
	 */
	final protected function get_assets_url($file_name, $file_extension)
	{
		$url = $this->get_assets_base_url() . $file_extension . '/' . $file_name;

		return $url . '.' . $file_extension;
	}

	/**
	 * Get js assets url
	 *
	 * @since 1.0.0
	 * @access protected
	 *
	 * @param string $file_name
	 *
	 * @return string
	 */
	final protected function get_js_assets_url($file_name)
	{
		return $this->get_assets_url($file_name, 'js');
	}

	/**
	 * Get css assets url
	 *
	 * @since 1.0.0
	 * @access protected
	 *
	 * @param string $file_name
	 *
	 * @return string
	 */
	final protected function get_css_assets_url($file_name)
	{
		return $this->get_assets_url($file_name, 'css');
	}

	/**
	 * Enqueue admin scripts.
	 *
	 * Registers all the admin scripts and enqueues them.
	 *
	 * Fired by `admin_enqueue_scripts` action.
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function enqueue_scripts()
	{
		wp_register_script(
			'custom-fields-manager' . '-admin',
			$this->get_js_assets_url('custom-fields-manager'),
			[],
			CUSTOM_FIELDS_MANAGER_VERSION,
			true
		);

		wp_enqueue_script('custom-fields-manager' . '-admin');
	}

	/**
	 * Enqueue admin styles.
	 *
	 * Registers all the admin styles and enqueues them.
	 *
	 * Fired by `admin_enqueue_scripts` action.
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function enqueue_styles()
	{
		wp_register_style(
			'custom-fields-manager' . '-admin',
			$this->get_css_assets_url('custom-fields-manager'),
			CUSTOM_FIELDS_MANAGER_VERSION
		);

		wp_enqueue_style('custom-fields-manager' . '-admin');
	}


	/**
	 * Plugin action links.
	 *
	 * Adds action links to the plugin list table
	 *
	 * Fired by `plugin_action_links` filter.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @param array $links An array of plugin action links.
	 *
	 * @return array An array of plugin action links.
	 */
	public function plugin_action_links($links)
	{
		$settings_link = sprintf('<a href="%1$s">%2$s</a>', admin_url('admin.php?page=' . 'custom-fields-manager'), esc_html__('Config', 'custom-fields-manager'));
		array_unshift($links, $settings_link);
		return $links;
	}

	/**
	 * Plugin row meta.
	 *
	 * Adds row meta links to the plugin list table
	 *
	 * Fired by `plugin_row_meta` filter.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @param array  $plugin_meta An array of the plugin's metadata, including
	 *                            the version, author, author URI, and plugin URI.
	 * @param string $plugin_file Path to the plugin file, relative to the plugins
	 *                            directory.
	 *
	 * @return array An array of plugin row meta links.
	 */
	public function plugin_row_meta($plugin_meta, $plugin_file)
	{
		if ('custom-fields-manager' === $plugin_file) {
			$row_meta = [
				'docs' => '<a href="' . CUSTOM_FIELDS_MANAGER__DOC_LINK . '" aria-label="' . esc_attr(esc_html__('View CFM Documentation', CUSTOM_FIELDS_MANAGER__FILE__)) . '" target="_blank">' . esc_html__('Docs & FAQs', CUSTOM_FIELDS_MANAGER__FILE__) . '</a>'
			];

			$plugin_meta = array_merge($plugin_meta, $row_meta);
		}


		return $plugin_meta;
	}

	/**
	 * Admin footer text.
	 *
	 * Modifies the "Thank you" text displayed in the admin footer.
	 *
	 * Fired by `admin_footer_text` filter.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @param string $footer_text The content that will be printed.
	 *
	 * @return string The content that will be printed.
	 */
	public function admin_footer_text($footer_text)
	{
		$current_screen = get_current_screen();
		$is_custom_fields_manager_screen = ($current_screen && false !== strpos($current_screen->id, 'custom-fields-manager'));
		if ($is_custom_fields_manager_screen) {
			$footer_text = sprintf(
				/* translators: 1: Custom Fields Manager, 2: Link to plugin review */
				__('Enjoyed %1$s? Please leave us a %2$s rating. We really appreciate your support!', 'custom-fields-manager'),
				'<strong>' . esc_html__(CUSTOM_FIELDS_MANAGER__TITLE, 'custom-fields-manager') . '</strong>',
				'<a href="' . CUSTOM_FIELDS_MANAGER__DOC_LINK . '" target="_blank">&#9733;&#9733;&#9733;&#9733;&#9733;</a>'
			);
		}
		return $footer_text;
	}

	/**
	 * Get custom-fields-manager dashboard overview widget footer actions.
	 *
	 * Retrieves the footer action links displayed in custom-fields-manager dashboard widget.
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public static function static_get_dashboard_overview_widget_footer_actions()
	{
		$base_actions = [
			'blog' => [
				'title' => esc_html__('Blog', 'custom-fields-manager'),
				'link' => CUSTOM_FIELDS_MANAGER__DOC_LINK,
			],
			'help' => [
				'title' => esc_html__('Help', 'custom-fields-manager'),
				'link' => CUSTOM_FIELDS_MANAGER__DOC_LINK,
			],
		];


		$actions = $base_actions;

		return $actions;
	}

	public function custom_fields_manager_check_plugin_version()
	{
		$plugin_data = get_plugin_data(CUSTOM_FIELDS_MANAGER__FILE__);
		$installed_version = $plugin_data['Version'];

		// Make a request to the API to fetch the latest version
		$api_url = CUSTOM_FIELDS_MANAGER__API_URL;
		$response = wp_remote_get($api_url);

		if (is_wp_error($response)) {
			return;
		}

		$api_data = json_decode(wp_remote_retrieve_body($response));
		if (!$api_data || !isset($api_data->version)) {
			return;
		}

		$latest_version = $api_data->version;

		if (version_compare($installed_version, $latest_version, '<')) {
			$message = sprintf(
				/* translators: 1: Plugin name, 2: Installed version, 3: Latest version */
				esc_html__('Warning: %1$s is outdated. You are using version %2$s, but the latest version is %3$s.', 'custom_fields_manager'),
				CUSTOM_FIELDS_MANAGER__TITLE,
				$installed_version,
				$latest_version
			);
			$plugin_site_link = sprintf(
				'<a href="%1$s" target="_blank">%2$s</a>',
				CUSTOM_FIELDS_MANAGER__DOC_LINK,
				esc_html__('Visit plugin site', 'custom_fields_manager')
			);
			$message .= ' ' . sprintf(
				/* translators: 1: Plugin site link */
				esc_html__('Please update to the latest version to ensure compatibility and access new features. %1$s', 'custom_fields_manager'),
				$plugin_site_link
			);

			echo wp_kses_post('<div class="notice notice-warning is-dismissible"><p>' . $message . '</p></div>');
		}
	}

	/**
	 * Admin constructor.
	 *
	 * Initializing custom-fields-manager in WordPress admin.
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function __construct()
	{

		add_action('admin_enqueue_scripts', [$this, 'enqueue_scripts']);
		add_action('admin_enqueue_scripts', [$this, 'enqueue_styles']);

		add_filter('plugin_action_links_' . CUSTOM_FIELDS_MANAGER_PLUGIN_BASE, [$this, 'plugin_action_links']);
		add_filter('plugin_row_meta', [$this, 'plugin_row_meta'], 10, 2);

		add_filter('admin_footer_text', [$this, 'admin_footer_text']);

		add_action('admin_notices', [$this, 'custom_fields_manager_check_plugin_version']);
	}
}
